<?php

/**
 * template tag: displays the language switcher
 *
 * list of parameters accepted in $args:
 *
 * dropdown               => displays a dropdown if set to 1, defaults to 0
 * echo                   => echoes the the switcher if set to 1 (default)
 * hide_if_empty          => hides languages with no posts (or pages) if set to 1 (default)
 * show_flags             => shows flags if set to 1, defaults to 0
 * show_names             => shows languages names if set to 1 (default)
 * display_names_as       => wether to display the language name or code. valid options are 'slug' and 'name'
 * force_home             => forces linking to the home page is set to 1, defaults to 0
 * hide_if_no_translation => hides the link if there is no translation if set to 1, defaults to 0
 * hide_current           => hides the current language if set to 1, defaults to 0
 * post_id                => if not null, link to translations of post defined by post_id, defaults to null
 * raw                    => set this to true to build your own custom language switcher, defaults to 0
 *
 * @since 0.5
 *
 * @param array $args optional
 * @return null|string|array null if displaying, array if raw is requested, string otherwise
 */
function pll_the_languages($args = '')
{
	global $polylang;
	if ($polylang instanceof PLL_Frontend && !empty($polylang->links)) {
		$switcher = new PLL_Switcher;
		return $switcher->the_languages($polylang->links, $args);
	}
	return '';
}



/**
 * returns the current language
 *
 * @since 0.8.1
 *
 * @param string $field optional the language field to return 'name', 'locale', 'slug', defaults to 'object'
 * @return string the requested field for the current language
 */
function aitLangsGetCurrentLanguage($field = 'object')
{
	global $polylang;

	if($field != 'object'){
		return isset($polylang->curlang->$field) ? $polylang->curlang->$field : false;
	}else{
		return isset($polylang->curlang) ? $polylang->curlang : false;
	}
}



/**
 * returns the default language
 *
 * @since 1.0
 *
 * @param string $field optional the language field to return 'name', 'locale', 'slug', defaults to 'object'
 * @return string the requested field for the default language
 */
function aitLangsGetDefaultLanguage($field = 'object')
{
	global $polylang;

	if(isset($polylang->options['default_lang'])){
		$lang = $polylang->model->get_language($polylang->options['default_lang']);
		if($lang){
			if($field != 'object'){
				return isset($lang->$field) ? $lang->$field : false;
			}else{
				return $lang;
			}
		}else{
			return false;
		}
	}

	return false;
}



function aitLangsGetLangForFiltering()
{
	return get_user_meta(get_current_user_id(), 'ait_langs_content_filtering_lang', true);
}



/**
 * among the post and its translations, returns the id of the post which is in the language represented by $slug
 *
 * @since 0.5
 *
 * @param int $post_id post id
 * @param string $slug optional language code, defaults to current language
 * @return int post id of the translation if exists
 */
function pll_get_post($post_id, $slug = '')
{
	global $polylang;
	return isset($polylang) && ($slug = $slug ? $slug : aitLangsGetCurrentLanguage('slug')) ? $polylang->model->get_post($post_id, $slug) : null;
}



/**
 * among the term and its translations, returns the id of the term which is in the language represented by $slug
 *
 * @since 0.5
 *
 * @param int $term_id term id
 * @param string $slug optional language code, defaults to current language
 * @return int term id of the translation if exists
 */
function pll_get_term($term_id, $slug = '')
{
	global $polylang;
	return isset($polylang) && ($slug = $slug ? $slug : aitLangsGetCurrentLanguage('slug')) ? $polylang->model->get_term($term_id, $slug) : null;
}



/**
 * returns the home url in the current language
 *
 * @since 0.8
 *
 * @return string
 */
function pll_home_url()
{
	global $polylang;
	return $polylang instanceof PLL_Frontend &&  !empty($polylang->links) ? $polylang->links->get_home_url() : home_url('/');
}



/**
 * Registers a string for translation in the "strings translation" panel
 *
 * @since 0.6
 *
 * @param  string  $name      A unique name for the string
 * @param  string  $string    The string to register
 * @param  string  $context   Optional the group in which the string is registered, defaults to 'ait-languages'
 * @param  boolean $multiline Optional wether the string table should display a multiline textarea or a single line input, defaults to single line
 * @return void
 */
function aitLangsRegisterString($name, $string, $context = 'ait-languages', $multiline = false)
{
	global $polylang;
	if($polylang instanceof PLL_Admin && !empty($polylang->settings_page)){
		$polylang->settings_page->register_string($name, $string, $context, $multiline);
	}
}



/**
 * translates a string (previously registered with pll_register_string)
 *
 * @since 0.6
 *
 * @param string $string the string to translate
 * @return string the string translation in the current language
 */
function ait__($string)
{
	return __($string, 'pll_string');
}



/**
 * echoes a translated string (previously registered with pll_register_string)
 *
 * @since 0.6
 *
 * @param string $string the string to translate
 */
function ait_e($string)
{
	_e($string, 'pll_string');
}



/**
 * returns true if Polylang manages languages and translations for this post type
 *
 * @since 1.0.1
 *
 * @param string post type name
 * @return bool
 */
function pll_is_translated_post_type($post_type)
{
	global $polylang;
	return isset($polylang) && $polylang->model->is_translated_post_type($post_type);
}



/**
 * returns true if Polylang manages languages and translations for this taxonomy
 *
 * @since 1.0.1
 *
 * @param string taxonomy name
 * @return bool
 */
function pll_is_translated_taxonomy($tax)
{
	global $polylang;
	return isset($polylang) && $polylang->model->is_translated_taxonomy($tax);
}
