<?php

/*
 * a class for the Polylang settings pages
 *
 * @since 1.2
 */
class PLL_Settings {
	public $model, $options;
	protected $strings = array(); // strings to translate

	/*
	 * constructor
	 *
	 * @since 1.2
	 *
	 * @param object $model instance of PLL_Model
	 */
	public function __construct(&$model) {
		$this->model = &$model;
		$this->options = &$model->options;

		// adds screen options and the about box in the languages admin panel
		add_action('load-theme-admin_page_ait-languages',  array(&$this, 'load_page'));

		// saves per-page value in screen option
		add_filter('set-screen-option', create_function('$s, $o, $v', 'return $v;'), 10, 3);
	}

	/*
	 * adds screen options and the about box in the languages admin panel
	 *
	 * @since 0.9.5
	 */
	public function load_page() {

		if (!isset($_GET['tab']) || $_GET['tab'] == 'lang')
			add_screen_option('per_page', array('label' => __('Languages', 'ait-languages'), 'default' => 10, 'option' => 'pll_lang_per_page'));


		if (isset($_GET['tab']) && $_GET['tab'] == 'strings')
			add_screen_option('per_page', array('label' => __('Strings translations', 'ait-languages'), 'default' => 10, 'option' => 'pll_strings_per_page'));
	}

	/*
	 * diplays the 3 tabs pages: languages, strings translations, settings
	 * also manages user input for these pages
	 *
	 * @since 0.1
	 */
	public function languages_page() {
		// prepare the list of tabs
		$tabs = array('lang' => __('Languages','ait-languages'));

		// only if at least one language has been created
		if ($listlanguages = $this->model->get_languages_list()) {
			$tabs['strings'] = __('Strings translation','ait-languages');
			$tabs['settings'] = __('Settings', 'ait-languages');
		}

		$activeTab = !empty($_GET['tab']) ? $_GET['tab'] : 'lang';

		$adminPageFile = dirname(__FILE__) . "/view-admin-page-{$activeTab}.php";

		if(!file_exists($adminPageFile)){
			echo __('Cheatin&#8217; uh?');
			return;
		}

		switch($activeTab) {
			case 'lang':
				// prepare the list table of languages
				$list_table = new PLL_Table_Languages();
				$list_table->prepare_items($listlanguages);

				// error messages for data validation
				$errors[1] = __('Enter a valid WordPress locale', 'ait-languages');
				$errors[2] = __('The language code contains invalid characters', 'ait-languages');
				$errors[3] = __('The language code must be unique', 'ait-languages');
				$errors[4] = __('The language must have a name', 'ait-languages');
				$errors[5] = __('The language was created, but the WordPress language file was not downloaded. Please install it manually.', 'ait-languages');
				break;

			case 'strings':
				// get the strings to translate
				$data = $this->get_strings();

				$selected = empty($_REQUEST['group']) ? -1 : $_REQUEST['group'];
				foreach ($data as $key=>$row) {
					$groups[] = $row['context']; // get the groups

					// filter for search string
					if (($selected !=-1 && $row['context'] != $selected) || (!empty($_REQUEST['s']) && stripos($row['name'], $_REQUEST['s']) === false && stripos($row['string'], $_REQUEST['s']) === false))
						unset ($data[$key]);
				}

				$groups = array_unique($groups);

				// load translations
				foreach ($listlanguages as $language){
					// filters by language if requested
					if(($lg = get_user_meta(get_current_user_id(), 'ait_langs_content_filtering_lang', true)) && $language->slug != $lg){
						continue;
					}

					$mo = new PLL_MO();
					$mo->import_from_db($language);

					foreach ($data as $key=>$row) {
						$data[$key]['translations'][$language->name] = $mo->translate($row['string']);
						$data[$key]['row'] = $key; // store the row number for convenience
					}
				}

				$string_table = new PLL_Table_String($groups, $selected);
				$string_table->prepare_items($data);
				break;

			case 'settings':

				$aitCpts = array_values(get_post_types(array('ait-translatable-cpt' => true)));
				$aitTaxs = array_values(get_taxonomies(array('ait-translatable-tax' => true)));

				$wcCpts = array('product', 'product_variation', 'shop_order', 'shop_coupon', 'shop_order_refund', 'shop_webhook'); // exclude woocommerce CPTs
				$wcTaxs = array('product_type', 'product_cat', 'product_tag', 'product_shipping_class', 'shop_order_status'); // exclude woocommerce taxonomies

				global $wc_product_attributes;

				if(isset($wc_product_attributes) and !empty($wc_product_attributes)){
					$wcTaxs = array_merge($wcTaxs, array_keys($wc_product_attributes));
				}

				$post_types = array_unique(apply_filters('pll_get_post_types',
					array_diff(
						get_post_types(array('_builtin' => false, 'show_ui' => false)),
						get_post_types(array('_pll' => true)),
						$aitCpts,
						$wcCpts
					), true));

				$taxonomies = array_unique(apply_filters('pll_get_taxonomies',
					array_diff(
						get_taxonomies(array('_builtin' => false, 'show_ui' => false)),
						get_taxonomies(array('_pll' => true)),
						$aitTaxs,
						$wcTaxs
					), true));

				break;

			default:
				break;
		}

		$using_permalinks = $GLOBALS['wp_rewrite']->using_permalinks();

		$action = isset($_REQUEST['pll_action']) ? $_REQUEST['pll_action'] : '';

		switch ($action) {
			case 'add':
				check_admin_referer( 'add-lang', '_wpnonce_add-lang' );

				$error = $this->model->add_language($_POST);

				if (0 == $error && !PLL_Admin::download_mo($_POST['locale']))
					$error = 5;

				wp_redirect(admin_url('admin.php?page=ait-languages' . ($error ? '&error=' . $error : ''))); // to refresh the page (possible thanks to the $_GET['noheader']=true)
				exit;
				break;

			case 'delete':
				check_admin_referer('delete-lang');

				if (!empty($_GET['lang']))
					$this->model->delete_language((int) $_GET['lang']);

				wp_redirect(admin_url('admin.php?page=ait-languages')); // to refresh the page (possible thanks to the $_GET['noheader']=true)
				exit;
				break;

			case 'edit':
				if (!empty($_GET['lang']))
					$edit_lang = $this->model->get_language((int) $_GET['lang']);

				break;

			case 'update':
				check_admin_referer( 'add-lang', '_wpnonce_add-lang' );

				$error = $this->model->update_language($_POST);

				wp_redirect(admin_url('admin.php?page=ait-languages' . ($error ? '&error=' . $error : ''))); // to refresh the page (possible thanks to the $_GET['noheader']=true)
				exit;
				break;

			case 'string-translation':
				if (!empty($_REQUEST['submit'])) {
					check_admin_referer( 'string-translation', '_wpnonce_string-translation' );
					$strings = $this->get_strings();

					foreach ($this->model->get_languages_list() as $language) {
						if(empty($_POST['translation'][$language->name])) // in case the language filter is active (thanks to John P. Bloch)
							continue;

						$mo = new PLL_MO();
						$mo->import_from_db($language);

						foreach ($_POST['translation'][$language->name] as $key=>$translation)
							$mo->add_entry($mo->make_entry($strings[$key]['string'], stripslashes($translation)));

						// clean database (removes all strings which were registered some day but are no more)
						if (!empty($_POST['clean'])) {
							$new_mo = new PLL_MO();

							foreach ($strings as $string)
								$new_mo->add_entry($mo->make_entry($string['string'], $mo->translate($string['string'])));
						}

						isset($new_mo) ? $new_mo->export_to_db($language) : $mo->export_to_db($language);
					}
				}

				do_action('pll_save_strings_translations');

				// unregisters strings registered through WPML API
				if ($string_table->current_action() == 'delete' && !empty($_REQUEST['strings']) && function_exists('icl_unregister_string')) {
					check_admin_referer( 'string-translation', '_wpnonce_string-translation' );
					$strings = $this->get_strings();

					foreach ($_REQUEST['strings'] as $key)
						icl_unregister_string($strings[$key]['context'], $strings[$key]['name']);
				}

				// to refresh the page (possible thanks to the $_GET['noheader']=true)
				$url = admin_url('admin.php?page=ait-languages&tab=strings');
				foreach(array('s', 'paged', 'group') as $qv){
					$url = empty($_REQUEST[$qv]) ? $url : $url . '&' . $qv . '=' . $_REQUEST[$qv];
				}
				wp_redirect($url);
				exit;
				break;

			case 'options':
				check_admin_referer( 'options-lang', '_wpnonce_options-lang' );

				$this->options['default_lang'] = sanitize_title($_POST['default_lang']); // we have slug as value

				foreach(array('force_lang', 'rewrite') as $key)
					$this->options[$key] = isset($_POST[$key]) ? (int) $_POST[$key] : 0;

				// FIXME : TODO error message if not a valid url
				if (isset($_POST['domains']) && is_array($_POST['domains']))
					foreach ($_POST['domains'] as $key => $domain)
						$this->options['domains'][$key] = esc_url_raw(trim($domain));

				foreach (array('browser', 'hide_default', 'redirect_lang', 'media_support') as $key)
					$this->options[$key] = isset($_POST[$key]) ? 1 : 0;

				foreach (array('sync', 'post_types', 'taxonomies') as $key)
					$this->options[$key] = empty($_POST[$key]) ? array() : array_keys($_POST[$key], 1);

				update_option('polylang', $this->options);

				// refresh rewrite rules in case rewrite,  hide_default, post types or taxonomies options have been modified
				// it seems useless to refresh permastruct here
				flush_rewrite_rules();

				// fills existing posts & terms with default language
				if (isset($_POST['fill_languages'])){
					$this->model->fillExistingPostsTermsWithDefaultLanguage();
				}
				wp_redirect(admin_url('admin.php?page=ait-languages&tab=settings&updated=true')); // updated=true interpreted by WP
				exit;
				break;

			default:
				break;
		}

		// displays the page
		include(PLL_ADMIN_INC . '/view-admin-layout.php');
	}

	/*
	 * register strings for translation making sure it is not duplicate or empty
	 *
	 * @since 0.6
	 *
	 * @param string $name a unique name for the string
	 * @param string $string the string to register
	 * @param string $context optional the group in which the string is registered, defaults to 'ait-languages'
	 * @param bool $multiline optional wether the string table should display a multiline textarea or a single line input, defaults to single line
	 */
	public function register_string($name, $string, $context = 'ait-languages', $multiline = false) {
		// backward compatibility with Polylang older than 1.1
		if (is_bool($context)) {
			$multiline = $context;
			$context = 'ait-languages';
		}

		$to_register = compact('name', 'string', 'context', 'multiline');
		if (!in_array($to_register, $this->strings) && $to_register['string'])
			$this->strings[] = $to_register;
	}

	/*
	 * get registered strings
	 *
	 * @since 0.6.1
	 *
	 * @return array list of all registered strings
	 */
	public function &get_strings() {
		// WP strings
		$this->register_string(__('Site Title'), get_option('blogname'), 'WordPress');
		$this->register_string(__('Tagline'), get_option('blogdescription'), 'WordPress');
		$this->register_string(__('Date Format'), get_option('date_format'), 'WordPress');
		$this->register_string(__('Time Format'), get_option('time_format'), 'WordPress');

		// widgets titles
		global $wp_registered_widgets;
		$sidebars = wp_get_sidebars_widgets();
		foreach ($sidebars as $sidebar => $widgets) {
			if ($sidebar == 'wp_inactive_widgets' || !isset($widgets))
				continue;

			foreach ($widgets as $widget) {
				// nothing can be done if the widget is created using pre WP2.8 API :(
				// there is no object, so we can't access it to get the widget options
				if (!isset($wp_registered_widgets[$widget]['callback'][0]) || !is_object($wp_registered_widgets[$widget]['callback'][0]) || !method_exists($wp_registered_widgets[$widget]['callback'][0], 'get_settings'))
					continue;

				$widget_settings = $wp_registered_widgets[$widget]['callback'][0]->get_settings();
				$number = $wp_registered_widgets[$widget]['params'][0]['number'];
				// don't enable widget title translation if the widget is visible in only one language or if there is no title
				if (empty($this->options['widgets'][$widget]) && isset($widget_settings[$number]['title']) && $title = $widget_settings[$number]['title'])
					$this->register_string(__('Widget title', 'ait-languages'), $title, 'Widget');
			}
		}

		// allow plugins to modify our list of strings, mainly for use by our PLL_WPML_Compat class
		$this->strings = apply_filters('pll_get_strings', $this->strings);
		return $this->strings;
	}

	/*
	 * list the post metas to synchronize
	 *
	 * @since 1.0
	 *
	 * @return array
	 */
	static public function list_metas_to_sync() {
		return array(
			'taxonomies'        => __('Taxonomies', 'ait-languages'),
			'post_meta'         => __('Custom fields', 'ait-languages'),
			'comment_status'    => __('Comment status', 'ait-languages'),
			'ping_status'       => __('Ping status', 'ait-languages'),
			'sticky_posts'      => __('Sticky posts', 'ait-languages'),
			'post_date'         => __('Published date', 'ait-languages'),
			'post_format'       => __('Post format', 'ait-languages'),
			'post_parent'       => __('Page parent', 'ait-languages'),
			'_wp_page_template' => __('Page template', 'ait-languages'),
			'menu_order'        => __('Page order', 'ait-languages'),
			'_thumbnail_id'     => __('Featured image', 'ait-languages'),
		);
	}
}
